# Presentation: https://docs.google.com/presentation/d/10j8hVtzN-l4JtiaV6tY60dXX3rRy5PwOQNK26h4w5yE/edit?usp=sharing

# Applets
# - Linear Regression: https://mathlets.org/mathlets/linear-regression/
# - Confidence Intervals: https://mathlets.org/mathlets/confidence-intervals/
# - Hypothesis Testing: https://www.stapplet.com/power.html

# Videos if you are still confused on previous topics:
# - Z Score, Confidence Interval, and Margin of Error: https://www.youtube.com/watch?v=DT-fPG0Hff8
# - Random walks: https://youtu.be/stgYW6M5o4k?si=WqGAkux5KWD1bzEx
# - OOP: https://youtu.be/JeznW_7DlB0?si=IpBOWv4lsZL1CUl7

import numpy as np
import matplotlib.pyplot as plt

# "True" population: pretend this is everyone
rng = np.random.default_rng(0)
population = rng.normal(loc=50, scale=10, size=100_000)  # mean 50, sd 10

# we draw many samples and compute the sample mean each time
n = 30 # sample size
num_samples = 2000
# we collect the sample means here
sample_means = []
for _ in range(num_samples):
    sample = rng.choice(population, size=n, replace=True)
    sample_means.append(np.mean(sample))

sample_means = np.array(sample_means)

# Standard Error (SE): how spread out the sample mean varies across possible samples

# empirical standard error: SD of sample means
# Simulation-based approximation of SE
empirical_se = np.std(sample_means)

# True SE ≈ population_sd / sqrt(n)
pop_sd = np.std(population) #aka σ
theoretical_se = pop_sd / np.sqrt(n) # why do we divide by sqrt(n) here?

# print(f"Population mean ≈ {np.mean(population):.2f}")
# print(f"Population SD ≈ {pop_sd:.2f}")
# print(f"Empirical SE of sample means ≈ {empirical_se:.2f}")
# print(f"Theoretical SE (SD/sqrt(n)) ≈ {theoretical_se:.2f}")

# code to plot the sample means
plt.hist(sample_means, bins=30, density=True)
plt.axvline(np.mean(population), linestyle="--")
plt.xlabel("Sample mean")
plt.ylabel("Density")
plt.title(f"Sampling distribution of mean (n={n})")
# uncomment to show plot
# plt.show()



# imagine we don't see the whole population, just one sample
population = rng.normal(loc=40, scale=15, size=100_000)
sample = rng.choice(population, size=50, replace=False)
sample_mean = np.mean(sample)
sample_sd = np.std(sample, ddof=1)
n = len(sample)

# In real life, we don't know the true standard deviation of the population
# So we can estimate it using the standard deviation of one sample:
# standard error of the mean
se = sample_sd / np.sqrt(n)


# 95% confidence interval using z ≈ 1.96
# z score: number of standard deviations from the mean needed to define the boundaries of the interval for a given confidence
z = 1.96
ci_lower = sample_mean - z * se
ci_upper = sample_mean + z * se

# print(f"Sample mean: {sample_mean:.2f}")
# print(f"Sample SD: {sample_sd:.2f}")
# print(f"Standard error: {se:.2f}")
# print(f"95% CI for mean: ({ci_lower:.2f}, {ci_upper:.2f})")

# how do confidence intervals work?
# what do confidence intervals say about the data?
def compute_ci(sample, z=1.96):
    m = np.mean(sample)
    s = np.std(sample, ddof=1)
    n = len(sample)
    se = s / np.sqrt(n)
    return m - z*se, m + z*se

true_mean = np.mean(population)

num_intervals = 200
count_contains = 0

for _ in range(num_intervals):
    sample = rng.choice(population, size=50, replace=False)
    low, high = compute_ci(sample)
    if low <= true_mean <= high:
        count_contains += 1

# print(f"{count_contains}/{num_intervals} intervals contained the true mean "
#       f"≈ {100*count_contains/num_intervals:.1f}%")



rng = np.random.default_rng(2)

# given 65 heads flipped out of 100 what does this say about whether the coin is fair?
# observed data
n_flips = 100
observed_heads = 65
observed_prop = observed_heads / n_flips

# print(f"Observed: {observed_heads} heads out of {n_flips} flips (p̂ = {observed_prop:.2f})")

np.random.seed(0)
# null hypothesis: fair coin, p = 0.5
p0 = 0.5
num_sims = 50_000 # number of simulated experiments under H0

sim_props = []
for _ in range(num_sims):
    # simulate n_flips under H0
    # how does the binomial function model H0?
    flips = rng.binomial(n=1, p=p0, size=n_flips)
    sim_props.append(np.mean(flips))

sim_props = np.array(sim_props)

# two-sided p-value: probability of seeing a proportion at least this far from 0.5
# original
dist_from_null_obs = abs(observed_prop - p0)
dist_from_null_sims = abs(sim_props - p0)

p_value = np.mean(dist_from_null_sims >= dist_from_null_obs)

# print(f"Approximate two-sided p-value: {p_value:.4f}")
# what does failing to reject H0 mean? does that mean we are accepting H0?
alpha = 0.05
if p_value < alpha:
    # print(f"p < {alpha} -> reject H0 (coin looks biased)")
    pass
else:
    # print(f"p >= {alpha} -> fail to reject H0 (no strong evidence of bias)")
    pass
