# recitation 11: dynamic programming
# 0/1 and complementary knapsack problem
# tabular DP code: full solutions vs. solution reconstruction from values
# floyd-warshall (optional)
# edit distance with memoization and tabular DP

# 1. complementary knapsack
"""

Standard knapsack:
    maximize total value under weight ≤ W

Complementary knapsack:
    achieve at least some VALUE threshold while minimizing COST.

Why do we do this?
    Sometimes constraints appear “in reverse”:
         - "We must get *at least* X benefit"
         - "We want to minimize cost/votes/risk/etc."

Traditional DP indexes by weight W.
Complementary knapsack indexes by VALUE, and stores minimal COST.
"""


def election_flipping_example():
    """
    ELECTION FLIPPING:
    ------------------
    Counties each have:
        - cost: number of votes required to flip the county
        - value: electoral votes gained

    Goal:
        Achieve at least EV_target with minimal total "cost".

    DP structure:
        dp[v] = minimum votes required to get *exactly* v electoral votes
    """
    EV_target = 10
    counties = [
        # (votes_to_flip, electoral_votes)
        (3000, 3),
        (5000, 4),
        (2000, 2),
        (10000, 8)
    ]

    # Maximum possible EV sum
    max_EV = sum(ev for _, ev in counties)
    INF = 10**15

    # Base DP array: dp[v] = minimal cost to reach v EV
    dp = [INF] * (max_EV + 1)
    dp[0] = 0

    # 1D knapsack DP (VALUE-indexed instead of weight-indexed!)
    for cost, ev in counties:
        # iterate backwards to avoid reusing items
        for v in range(max_EV, ev - 1, -1):
            dp[v] = min(dp[v], dp[v - ev] + cost)

    best_cost = min(dp[EV_target:])
    print("--- Election Flipping (Complementary Knapsack) ---")
    print("Min votes required:", best_cost)
    print()

# another example of knapsack if you want/choose to go through it
def escape_room_complementary():
    """
    fun example: escape room bomb diffusal
    ---------------------------------------
    Each wire has:
        risk   = danger of cutting it (cost)
        safety = how much safer the bomb becomes (value)

    Goal:
        Achieve safety >= SAFETY_TARGET with MINIMUM risk.

    This is a good use-case for complementary knapsack.
    """

    wires = [
        (8, 5),   # red wire   --> risk=8, safety=5
        (3, 3),   # blue wire  --> risk=3, safety=3
        (6, 6),   # yellow wire
        (2, 2),   # green wire
        (10, 9),  # white wire
        (4, 4),   # black wire
        (7, 7),   # purple wire
    ]

    SAFETY_TARGET = 12
    max_safety = sum(s for _, s in wires)
    INF = 10**12

    dp = [INF] * (max_safety + 1)
    dp[0] = 0

    # value-indexed knapsack DP
    for risk, safety in wires:
        for s in range(max_safety, safety - 1, -1):
            dp[s] = min(dp[s], dp[s - safety] + risk)

    best_risk = min(dp[SAFETY_TARGET:])
    print("--- Escape Room Bomb Diffusal (Complementary Knapsack) ---")
    print("Minimum risk to reach safety target:", best_risk)
    print()


# 2. standard 0/1 knapsack - two coding implementations 
"""
here are two tabular-DP approaches:

(A) DP table stores FULL SOLUTIONS at each cell
    dp[i][w] = (best_value, [list of items])
    Pros: EASY to teach/visualize
    Cons: memory-heavy (O(nW) lists)

(B) DP table stores only values.
    Then we reconstruct solution by walking backwards.
    Pros: industry standard, memory efficient
    Cons: requires reconstruction logic
"""

items = [
    (2, 3),  # (weight=2, value=3)
    (3, 4),
    (4, 5),
    (5, 8)
]
capacity = 7


# a. full-solution DP table
def knapsack_store_solutions(items, W):
    """
    dp[i][w] = (best value using first i items, list of items chosen)

    Teaching notes:
        This implementation is VERY literal—each DP cell stores the
        *actual best subset* for that subproblem.

        This is a great way to demonstrate:
            - what subproblems look like
            - that DP tables can store ANYTHING, not just numbers
    """
    pass


# b. value-only dp + reconstruction step
def knapsack_value_only(items, W):
    """
    dp[i][w] = best value using first i items and weight exactly w

    notes:
        - DP table stores *only numbers*
        - Reconstruction walks backwards to discover chosen items
        - This is exactly how Floyd–Warshall’s table is structured
    """
    pass


# 3. floyd-warshall (tabular DP) -- OPTIONAL 
"""
Key teaching insight:
    Floyd–Warshall's DP structure is almost identical to knapsack:

    dp[i][w]      <-- DP over (item, weight)
    dist[i][j]    <-- DP over (start node, end node)

    Update rule:
        dp[i][w] = ...
        dist[i][j] = min(dist[i][j], dist[i][k] + dist[k][j])

    Same idea:
        Fill a table using results of smaller subproblems
"""


def floyd_warshall_demo():
    pass
    # INF = 10**9

    # dist = [
    #     [0,   3,  INF],
    #     [INF, 0,   2 ],
    #     [1,  INF,  0 ]
    # ]

    # print("--- Floyd–Warshall Demo ---")
    # print("Initial distance matrix:")
    # for row in dist:
    #     print(row)
    # print()

    # n = len(dist)

    # # standard Floyd–Warshall triple loop
    # for k in range(n):
    #     for i in range(n):
    #         for j in range(n):
    #             dist[i][j] = min(dist[i][j], dist[i][k] + dist[k][j])

    # print("after Floyd–Warshall:")
    # for row in dist:
    #     print(row)
    # print()


# 4. edit distance
"""
edit distance (levenshtein distance)
------------------------------------
Allowed operations:
    - Insert
    - Delete
    - Substitute (0 cost if same character)

dp[i][j] = minimum edits to convert a[:i] to b[:j]

Two implementations:
    • Memoization (top-down)
    • Tabular (bottom-up)
"""

def edit_distance_memo(a, b):
    pass


def edit_distance_tabular(a, b):
    pass


# main execution
if __name__ == "__main__":

    # complementary knapsack examples
    election_flipping_example()
    escape_room_complementary()

    # standard knapsack examples
    print("--- knapsack: DP storing full solutions ---")
    val, sol = knapsack_store_solutions(items, capacity)
    print("best value:", val)
    print("items taken:", sol)
    print()

    print("--- knapsack: DP storing only values + reconstruction ---")
    val2, sol2 = knapsack_value_only(items, capacity)
    print("best value:", val2)
    print("items taken:", sol2)
    print()

    # Floyd–Warshall
    floyd_warshall_demo()

    # edit distance
    print("--- edit distance --- ")
    a, b = "kitten", "sitting"
    print("memoized:", edit_distance_memo(a, b))
    print("tabular:", edit_distance_tabular(a, b))
    print()