"""
- When can you use greedy?
- Why use enumeration?
- Why DP?
"""

def make_knapsack():
    names = ["gold", "saffron", "diamond"]
    densities = [2, 1, 3]
    weights = [1,2,1]
    values = [density * weight for density, weight in zip(densities, weights)]
    capacity = 3
    return names, values, weights, capacity

### No Memo Vaguely DP Pseudocode ###
    # base case: no more items, return empty solution
    # case 1: try to take first item
    #   case 1a: item's weight is at/under capacity: take item. 
    #            reduce capacity, and solve knapsack on remaining items
    #   case 1b: item's weight is over capacity: don't take item. go to case 2
    # case 2: don't take first item, solve knapsack on remaining items 
    #         and unchanged capacity
    # final decision: return max value across both cases

### Memo DP Pseudocode ###
    # lazy person solution: return if already in memo (solved for us)
    # base case: no more items, return empty solution
    # case 1: try to take first item
    #   case 1a: item's weight is at/under capacity: take item. 
    #            reduce capacity, and solve knapsack on remaining items
    #   case 1b: item's weight is over capacity: don't take item. go to case 2
    # case 2: don't take first item, solve knapsack on remaining items 
    #         and unchanged capacity
    # final decision: pick max value across both cases
    # store final decision for this case in memo

# Iterative approach
# define subproblem as table[i][cap] = max value using items[i:n] w/ remaining capacity cap
### Tabular DP Pseudocode ###
# Overall idea: build solutions to smaller subproblems first
    # build table that is (n+1) x (capacity +1) = 6x6, and initialize all vals to 0
    # Start w/ table[n-1][0] meaning no remaining capacity left and no itmes left to consider
    # base case is entire last row (all zero), where index = len(items)
    # case 1: get value if don't take item
    #   val_without =  table[i+1][cap], capacity unchanged
    # case 2: get value if take item (if weight <= remaining capacity)
    #   val_with = table[i+1][cap-weight] + value, update capacity since took this item
    # store value for table[i][cap] to be max(val_with, val_without)

    # now that we have a table of values, need to return optimal solution
    # values stay the same down one column only if it is optimal NOT to take an item
    # start scanning table from i in range(0,n) and full capacity = 5
    # case 1: if table[i][remaining] = table[i+1][remaining] (didn't take that item)
    # case 2: take item (append name to solutions), update remaining

### Knapsack Memoization Review
def knapsack_indexed_memoized(items, capacity):
    memo = {}
    return kim_helper(items, 0, capacity, memo)


def kim_helper(items, index, capacity, memo):
    # bypass if already in memo
    if (index, capacity) in memo:
        return memo[index, capacity]

    # base case: no more items, empty solution
    if index == len(items):
        return [], 0

    # recursion: decide whether to take first item, solve for remaining items
    name, value, weight, _ = items[index]

    sol_without, val_without = kim_helper(items, index + 1, capacity, memo)

    if weight <= capacity:
        sol_with, val_with = kim_helper(items, index + 1, capacity - weight, memo)
        sol_with = sol_with + [name]  # need to make new list, due to memoization
        val_with += value
    else:
        sol_with = []
        val_with = 0

    if val_with > val_without:
        answer = sol_with, val_with
    else:
        answer = sol_without, val_without
    memo[index, capacity] = answer
    return answer


### Knapsack tabular review
def knapsack_indexed_tabular_values(items, capacity):
    # initialize table of subproblem VALUES
    n = len(items)
    table = [[0 for _ in range(capacity + 1)] for _ in range(n + 1)]

    # base case is encoded in last row, where index == len(items)

    # fill in table with VALUES only
    for i in range(n - 1, -1, -1):
        name, value, weight, _ = items[i]

        for cap in range(capacity + 1):

            val_without = table[i + 1][cap]

            if weight <= cap:
                val_with = table[i + 1][cap - weight] + value
            else:
                val_with = 0

            if val_with > val_without:
                table[i][cap] = val_with
            else:
                table[i][cap] = val_without

    # trace through values to find items in the optimal solution
    solution = []
    total_value = table[0][capacity]

    remaining = capacity
    for i in range(0, n):
        # values stay the same down one column only if optimal NOT to take an item
        if table[i][remaining] == table[i + 1][remaining]:
            continue
        else:
            name, value, weight, _ = items[i]
            solution.append(name)
            remaining -= weight

    return solution, total_value


names, values, weights, capacity = make_knapsack()
items = [(names[index], values[index], weights[index], capacity) for index in range(len(names))]
print(f"memoized results:\n {knapsack_indexed_memoized(items, capacity)}")
print(f"tabular results:\n {knapsack_indexed_tabular_values(items, capacity)}")


#########################################################################
# Exercise 1: Minimum edit distance between strings using memo
#########################################################################

def min_edit_distance_memo(str1, str2):
    """
    Computes the minimum edit distance between two strings using memoization.

    The edit distance is the minimum number of single-character operations
    required to transform str1 into str2. The allowed operations are:
        - insertion
        - deletion
        - replacement

    Parameters
    ----------
    str1 : str
        The original string we want to edit.
    str2 : str
        The target string we want str1 to match.

    Returns
    -------
    int
        The minimum number of edits required to convert str1 into str2.
    """
    memo = {}
    return min_edit_distance_memo_helper(str1, str2, 0, 0, memo)

def min_edit_distance_memo_helper(str1, str2, i, j, memo):
    """
    A recursive helper for min_edit_distance_memo() that computes the edit distance
    between the suffixes str1[i:] and str2[j:] using memoization.

    Parameters:
    str1 : str
        The original string
    str2 : str
        The target string
    i : int
        Current index in str1 (we are considering the substring str1[i:])
    j : int
        Current index in str2 (we are considering the substring str2[j:])
    memo : dict
        A dictionary used for memoization. It should map pairs (i, j) to the
        minimum number of edits needed to convert str1[i:] into str2[j:]

    Returns
    -------
    The minimum number of edits needed to transform str1[i:] into str2[j:].
    """
    # Check if result is already in memo
    if (i, j) in memo:
        return memo[i, j]

    # Base cases
    if i == len(str1):
        # If str1 is exhausted, we need to insert all remaining characters of str2
        return len(str2) - j
    if j == len(str2):
        # If str2 is exhausted, we need to delete all remaining characters of str1
        return len(str1) - i

    if str1[i] == str2[j]:
        # Characters match, move to the next pair
        result = min_edit_distance_memo_helper(str1, str2, i + 1, j + 1, memo)
    else:
        # Characters do not match, consider all three operations
        insert_cost = 1 + min_edit_distance_memo_helper(str1, str2, i, j + 1, memo)  # Insertion
        delete_cost = 1 + min_edit_distance_memo_helper(str1, str2, i + 1, j, memo)  # Deletion
        replace_cost = 1 + min_edit_distance_memo_helper(str1, str2, i + 1, j + 1, memo)  # Replacement

        result = min(insert_cost, delete_cost, replace_cost)

    # Store result in memo before returning
    memo[i, j] = result
    return result

# example usage
print(min_edit_distance_memo("abc", "abc"))  # Expected output: 0
print(min_edit_distance_memo("", "abc"))  # Expected output: 3
print(min_edit_distance_memo("kitten", "sitting"))  # Expected output: 3
print(min_edit_distance_memo("intention", "execution"))  # Expected output: 5

#########################################################################
# Exercise 2: Minimum edit distance between strings using tabulation
#########################################################################

def min_edit_distance_tabular(str1, str2):
    """
    Computes the minimum edit distance between two strings using memoization.

    The edit distance, parameters and returns are the same as defined in min_edit_distance_memo().
    """
    n = len(str1)
    m = len(str2)

    # Initialize the DP table
    table = [[0 for _ in range(m + 1)] for _ in range(n + 1)]

    # Base cases: transforming empty str1 to str2 and vice versa
    for i in range(n + 1):
        table[i][m] = n-i  # Deleting all characters from str1
    for j in range(m + 1):
        table[n][j] = m-j  # Inserting all characters to str1 to form str2

    # Fill the DP table from bottom up (reverse order)
    for i in range(n - 1, -1, -1):
        for j in range(m - 1, -1, -1):
            if str1[i] == str2[j]:
                table[i][j] = table[i + 1][j + 1]  # No operation needed
            else:
                insert_cost = 1 + table[i][j + 1]    # Insertion
                delete_cost = 1 + table[i + 1][j]    # Deletion
                replace_cost = 1 + table[i + 1][j + 1]  # Replacement

                table[i][j] = min(insert_cost, delete_cost, replace_cost)

    return table[0][0]

print(min_edit_distance_tabular("abc", "abc"))  # Expected output: 0
print(min_edit_distance_tabular("", "abc"))  # Expected output: 3
print(min_edit_distance_tabular("kitten", "sitting"))  # Expected output: 3
print(min_edit_distance_tabular("intention", "execution"))  # Expected output: 5
