"""

    [ cmd [mac]/ctrl[windows] + click to open ]

    - Some slides from sec1: https://docs.google.com/presentation/d/1mVW1SycCWBUIeE4zcL3dw_xHiL9ygZ9KJqN2BF93Lzk/edit?slide=id.p#slide=id.p
    - environment diagram visualization: https://pythontutor.com/python-compiler.html#mode=edit
    - Dijkstra's visualization: https://www.cs.usfca.edu/~galles/visualization/Dijkstra.html
    - DFS/BFS visualization: https://visualgo.net/en/dfsbfs

    TOPICS:

    - Graphs
        - DFS
        - BFS
        - Dijkstra's
    - Classes, Attributes, Methods, Inheritance
    - Monte Carlo simulations, probability (meh)
    - Random walks
        - List comprehension
        - Higher-order functions, lambdas
    - Environment diagrams (with classes!)
    - Exceptions

    """


############################################################
# Graph Helpers
############################################################


def neighbors(graph, node):
    return graph.get(node, [])

def path_to_string(path):
    """path is a list of nodes"""
    if path is None:
        return "None"
    node_strs = []
    for node in path:
        node_strs.append(str(node))
    return " --> ".join(node_strs)


############################################################
# Depth-first search - DFS
############################################################

"""
 - We implemented these using recursion and with a stack
 - Explores as far as possible down a branch before backtracking
 - Good for exploring possible paths, detecting cycles, finding patterns
 - DFS/BFS visualization: https://visualgo.net/en/dfsbfs

"""

def dfs_graph_helper(graph, goal, path):
    print("Current DFS path:", path_to_string(path))

    current_node = path[-1]
    if current_node == goal:
        return path

    for next_node in neighbors(graph, current_node):

        # avoid self-loops
        if next_node in path:
            print(f"  AVOID self-loop from {current_node} to {next_node}")
            continue

        result = dfs_graph_helper(graph, goal, path + [next_node])
        if result is not None:
            return result

    return None


def dfs_graph(graph, start, goal):
    return dfs_graph_helper(graph, goal, [start])


def demo_dfs_graph():
    flights = {
        "Boston": ["Providence", "New York"],
        "Providence": ["Boston", "New York"],
        "New York": ["Chicago"],
        "Chicago": ["Denver", "Phoenix"],
        "Denver": ["New York", "Phoenix"],
        "Los Angeles": ["Boston"],
    }

    # print(dfs_graph(flights, "Boston", "Phoenix"))
    # print()

    # change the ordering in which we explore Chicago's neighbors
    flights["Chicago"].reverse()

    # print(dfs_graph(flights, "Boston", "Phoenix"))
    # print()


# demo_dfs_graph()

"""
Stack implementation of DFS
"""

def dfs_lifo(graph, start, goal):
    stack = [[start]]

    while len(stack) > 0:
        print("Current stack:", pathlist_to_string(stack))

        # simulate running the body of a recursive dfs() call
        path = stack.pop(-1)
        print("  Current DFS path:", path_to_string(path))

        current_node = path[-1]
        if current_node == goal:
            return path

        # put children on stack in reverse order of which
        # we intend to explore them, because stack is lifo
        for next_node in reversed(neighbors(graph, current_node)):
            if next_node in path:
                continue

            # prepare to simulate running dfs() on next_node
            stack.append(path + [next_node])

    return None


############################################################
# Breadth-first Search - BFS
############################################################

"""
 - We implemented these using frontiers and queues
 - Explores neighbors level by level
 - Get current path + node from front (index 0)
    of queue and adds paths to current node's neighbors to end of queue
 - Finds the shortest path on UNWEIGHTED graphs

 - DFS/BFS visualization: https://visualgo.net/en/dfsbfs

"""

def pathlist_to_string(queue):
    """path is a list of nodes"""
    path_strs = []
    for path in queue:
        path_strs.append(path_to_string(path))
    return "[" + ", ".join(path_strs) + "]"

def bfs_graph(graph, start, goal):
    current_frontier = [[start]]
    next_frontier = []
    visited = {start}

    while len(current_frontier) > 0:
        print("Current frontier:", pathlist_to_string(current_frontier))

        for path in current_frontier:
            print("  Current BFS path:", path_to_string(path))

            current_node = path[-1]
            if current_node == goal:
                return path

            for next_node in neighbors(graph, current_node):

                # avoid nodes already seen in current or previous frontier
                if next_node in visited:
                    print(f"    AVOID revisiting {next_node}")
                    continue
                visited.add(next_node)
                next_frontier.append(path + [next_node])

        current_frontier, next_frontier = next_frontier, []

    return None


def demo_bfs_graph():
    flights = {
        "Boston": ["Providence", "New York"],
        "Providence": ["Boston", "New York"],
        "New York": ["Chicago"],
        "Chicago": ["Denver", "Phoenix"],
        "Denver": ["New York", "Phoenix"],
        "Los Angeles": ["Boston"],
    }

    # print(bfs_graph(flights, "Boston", "Phoenix"))
    # print()

    # change the ordering in which we explore Chicago's neighbors
    flights["Chicago"].reverse()

    # shortest path solution does not change
    # print(bfs_graph(flights, "Boston", "Phoenix"))
    # print()


# demo_bfs_graph()


"""
Queue-based implementation of BFS
"""

def bfs_fifo(graph, start, goal):
    if start == goal:
        return [start]
    queue = [[start]]
    visited = {start}

    while len(queue) > 0:
        print("Current queue:", pathlist_to_string(queue))

        # simulate iterating through current frontier
        path = queue.pop(0)
        print("  Current BFS path:", path_to_string(path))

        current_node = path[-1]
        for next_node in neighbors(graph, current_node):
            if next_node in visited:
                continue
            visited.add(next_node)
            new_path = path + [next_node]
            if next_node == goal:
                return new_path

            # simulate building next frontier
            queue.append(new_path)
    return None


############################################################
# Dijkstra's Algorithm
############################################################

"""
 - Find shortest paths on WEIGHTED graphs
 - Implemented using a priority queue:
    - Store distance from start node along with paths
    - Pop shortest cost path from the priority queue for our current path + node
 - We used it to find shortest paths to a target
 - Dijkstra's visualization: https://www.cs.usfca.edu/~galles/visualization/Dijkstra.html
    - Visualization finds shortest paths to all nodes using Dijkstra's. In class we stop when
      we pop the path to the current node from our priority queue
"""

def remove_min(queue):
    assert len(queue) > 0
    best_idx = 0
    min_so_far = queue[0]
    for idx in range(1, len(queue)):
        item = queue[idx]
        if item < min_so_far:
            best_idx = idx
            min_so_far = item
    return queue.pop(best_idx)


def find_node(queue, node):
    for idx in range(len(queue)):
        (cost, path) = queue[idx]
        if path[-1] == node:
            return idx
    return None


def update_node(queue, node, new_cost, new_path):
    idx = find_node(queue, node)
    if idx is None:
        print(f"  Adding path to {node!r}")
        queue.append((new_cost, new_path))
        print(f"  Current queue: {queue}")
    else:
        old_cost, old_path = queue[idx]
        if new_cost < old_cost:
            print(f"  Updating path to {node!r}")
            queue[idx] = (new_cost, new_path)
            print(f"  Current queue: {queue}")
        else:
            print(f"  No change to queue as new cost ({new_cost}) is not smaller ({old_cost})")


def dijkstra(graph, start, goal):
    # store best cost and path for discovered nodes that are
    # on present or future frontier
    queue = [(0, [start])]
    # separately, store nodes that are on past frontiers
    finished = set()

    while len(queue) > 0:
        print(f"Current queue: {queue}")

        # get a path off the true frontier
        cost, path = remove_min(queue)
        current_node = path[-1]
        finished.add(current_node)
        print(f"  Finished {current_node!r} with cost {cost}. Finished queue : {finished}")

        # optimality guaranteed for current node, return if goal
        if current_node == goal:
            return (cost, path)

        print(f"Expanding to neighbors from: {current_node}")
        # update paths to neighbors on priority queue
        for edge in neighbors(graph, current_node):
            (next_node, weight) = edge
            if next_node not in finished:
                print(f" Processing {current_node!r}-->{next_node!r} with weight {weight}")
                new_cost = cost + weight
                new_path = path + [next_node]
                update_node(queue, next_node, new_cost, new_path)

        print()

    return None


def demo_dijkstra():
    flights = {
        "Boston": [("Providence", 1), ("New York", 2)],
        "Providence": [("Boston", 1), ("New York", 2)],
        "New York": [("Chicago", 3)],
        "Chicago": [("Phoenix", 5), ("Denver", 3)],
        "Denver": [("New York", 4), ("Phoenix", 1)],
    }
    print(dijkstra(flights, "Boston", "Phoenix"))


# demo_dijkstra()

############################################################
# Monte Carlo Simulations
############################################################

"""
 - Introduced random module
    - random.seed(a=None)
    - random.random() → float in [0.0, 1.0)
    - random.uniform(a, b) → float in [a, b]
    - random.randint(a, b) → integer in [a, b] (inclusive)
    - random.choice(seq) → random element from a sequence
    - random.choices(seq, weights=None, k=1) → list of k elements (with optional weighting)
    - random.sample(population, k) → unique sample (no repeats)
    - random.shuffle(seq) → shuffles list in place

"""

import random

# Estimating pi example:
def throw_needles(num_needles):
    in_circle = 0
    for _ in range(num_needles):
        x = random.random()
        y = random.random()
        if (x*x + y*y)**0.5 <= 1:
            in_circle += 1
    # Counting needles in one quadrant only, so multiply by 4
    return 4 * in_circle / num_needles


# random.seed(3333)
# print("Estimating pi (3.14159) with increasing number of needles:")
# for p in range(1, 8):
#     n = 10**p
#     est = throw_needles(n)
#     print("With ", n, "needles, estimate for pi:", est)


############################################################
# List Comprehension
############################################################

"""
 - Shorter syntax for creating a list from values of an iterable
 - new_list = [expression for item in iterable]
     - expression: This is the operation performed on each item from the iterable.
                    The result of this expression is what will be added to the new_list
     - item: This is the variable that takes on each value from the iterable during the iteration
     - iterable: This is any object that can be iterated over (e.g., a list, tuple, string, range, etc.)
 - Can also filter for items that meet a condition, only accept items when condition is True:
    - new_list = [expression for item in iterable if condition]

    https://www.w3schools.com/python/python_lists_comprehension.asp

"""

# Longer syntax for l
# l = []
# for n in range(20):
#     if n%2==1:
#         l.append(n)

l = [n for n in range(20) if n%2==1]
l2 = [n**2 for n in l]
l3 = [n**2 for n in range(20) if  n%2==1]

# print(l3)

############################################################
# Lambdas
############################################################
"""

 - Short way to create a function in a single line
 - Creates a function object without naming it/assigning to a name
 - Convienient when one-line functions needed as parameters
    - Ex. key function in .sort(), sorted()
 - Syntax: lambda arguments : expression
    - Accepts arguments, and returns expression

 https://www.w3schools.com/python/python_lambda.asp

"""

square = lambda x : x**2
print(square(5)) # output 25

pow = lambda base, exp : base ** exp
print(pow(3,5)) # output 243


############################################################
# Higher-order functions
############################################################
"""
 - Functions are objects, and thus can be passed to functions as parameters
    or returned by functions
"""


############################################################
# Random Walks
############################################################
