import random

##################################################
# Stochastic Thinking: Applications of Randomness
##################################################

# Examples of stochastic processes
# - Stochastic gradient descent
# - Temperature parameter in LLMs
# - Randomized algorithms
# - Estimation and Monte Carlo methods

##############################
# Probabilistic Overview
##############################

# 1. If you roll 2 dice, what is the likelihood of getting a sum of 10? Answer: 1/12
# 2. If you roll 2 dice, what is the probability that at least 1 dice is a 5? Answer: 11/36

################################
# Probability Code Practice
################################

random.seed(0)

def biased_coin_flip(p):
    """
    Simulates a biased coin flip with a given probability of heads (p).

    Args:
        p: The probability of getting heads (a float between 0 and 1).
    Returns:
        'Heads' if the coin lands heads with probability p, 'Tails' otherwise.
    """
    #random.random() --> gives random # between 0 (inclusive) and 1 (exclusive)
    if random.random() < p:
        return 'Heads'
    else:
        return 'Tails'

def sample_uniform(a,b):
    '''
    Implement random.uniform(a,b)

    return a random float in the interval [a,b] where b > a over a uniform distribution
    '''
    return random.random()*(b-a) + a

################################
# Intro Monte Carlo
################################

# Rather than listing out all the 6^7 possible outcomes, we can estimate the probability
# by randomly sampling 7 dice (as an outcome) repeatedly and see if that outcome is favorable or not.

# Increasing number of trials increases accuracy of our estimate
def simulate_dice_probability(num_trials):
    """
    Use a simulation to estimate the probability that the sum of 7 dice is divisible by either 5 or 8
    """
    favorable_outcomes = 0
    for _ in range(num_trials):
        dice_sum = 0
        #for each trial roll 7 dice and get collective sum
        for _ in range(7):
            dice_sum += random.randint(1, 6) #for each dice pick a random value from 1-6 (inclusive)
        #check if that sum is favorable or not
        if dice_sum % 5 == 0 or dice_sum % 8 == 0:
            favorable_outcomes += 1
    return favorable_outcomes / num_trials

# answer should be ~0.325
def demo_dice_simulation():
    for num_trials in range(1, 100, 5):
        print(f"When num_trials={num_trials}, probability={simulate_dice_probability(num_trials)}")

demo_dice_simulation()

##################################################
# Monte Carlo Integration
##################################################

# Sample function to integrate over

def cubic(x):
    return x**3 - 2*x + 1

# Monte Carlo Integration function 1

# Why does f(x) need to be >= 0 on all x?:
# Otherwise we would overcount the area where f(x) < 0. 
# We would need to separately subtract off the area under the curve and below the x-axis.
def integrate_monte_carlo_bbox(f, a, b, num_needles):
    '''
    Approximates the integral of f over the interval [a,b], where f(x) >=0 for all x
    1. approximate y_max by scanning over [a,b]
    2. simulate throwing num_needles needles in a bounding box
    3. return approximate area
    '''
    y_max = 0
    epsilon = 1e-3
    
    try_x = a
    while try_x <= b:
        y_max = max(y_max, f(try_x))
        try_x += epsilon

    needles_under_curve = 0
    for i in range(num_needles):
        x = sample_uniform(a,b)
        y = sample_uniform(0,y_max)

        if y <= f(x):
            needles_under_curve += 1

    total_area = (b-a)*y_max
    return total_area*(needles_under_curve/num_needles)

# what would happen if we increase subdivisions?
# We get a more precise estimate of area under the curve
# what happens when we increase num_needles?
# We get a more precise estimate of area under the curve
def integrate(f, a, b, subdivisions = 5, num_needles = 1000):
    interval_width = (b - a) / subdivisions
    current_a = a
    current_b = a + interval_width
    integral = 0

    while current_b <= b:
        integral += integrate_monte_carlo_bbox(f, current_a, current_b, num_needles)
        current_a += interval_width
        current_b = current_a + interval_width
    return integral 


def demo_integrate():
    # actual integral is 2.75
    a = -1.5
    b = 0.5
    num_trials = 10
    for subdivisions in range(1, 15):
        final_integrate = 0
        # why do we have trials here?
        # To account for randomness (heavy over and underestimates), allows us to get a more accurate estimate
        for trial in range(0, num_trials):
            final_integrate += integrate(cubic, a, b, subdivisions)
        final_integrate /= num_trials
        print(f"integrating over [{a}, {b}] with {subdivisions} subdivisions: {final_integrate}")


demo_integrate()