import time
import os
import concurrent.futures
import cProfile
import pstats
from concurrent.futures import ThreadPoolExecutor, ProcessPoolExecutor

####################################
# Python byte code
####################################

import dis

def foo():
    x = 10
    y = 20
    return x * y

# dis.dis(foo)

####################################
# Profiler example
####################################



def slow_function():
    total = 0
    for i in range(1000000):
        total += i
    return total

def medium_function():
    time.sleep(0.2)

def fast_function():
    return sum(range(1000))

def main():
    slow_function()
    medium_function()
    fast_function()

# profiler = cProfile.Profile()
# profiler.runcall(main)
# stats = pstats.Stats(profiler)
# stats.sort_stats("cumtime")  # sort by total time spent in function
# stats.print_stats()


####################################
# Visual profiler example
####################################

# profiler = cProfile.Profile()
# profiler.enable()
# main()
# profiler.disable()

# # write the profile data to disk
# profiler.dump_stats("example.prof")
# print("Wrote profile to example.prof")

####################################
# Parrallelism benchmark CPU-bound
####################################

def fib(n):
    if n < 2:
        return n
    return fib(n - 1) + fib(n - 2)


def time_sequential(task_list):
    start = time.perf_counter()
    results = [fib(x) for x in task_list]
    return time.perf_counter() - start, results

def time_threadpool(task_list, max_workers):
    start = time.perf_counter()
    with concurrent.futures.ThreadPoolExecutor(max_workers=max_workers) as exe:
        results = list(exe.map(fib, task_list))
    return time.perf_counter() - start, results

def time_processpool(task_list, max_workers):
    start = time.perf_counter()
    with concurrent.futures.ProcessPoolExecutor(max_workers=max_workers) as exe:
        results = list(exe.map(fib, task_list))
    return time.perf_counter() - start, results


# if __name__ == "__main__":
#     n = 35            # Fibonacci number to compute (CPU-heavy)
#     tasks = 8         # Number of tasks
#     task_list = [n] * tasks

#     print(f"Running fib({n}) {tasks} times...\n")

#     # Sequential
#     seq_time, seq_res = time_sequential(task_list)
#     print(f"Sequential: {seq_time:.4f} seconds")

#     # Threading
#     thread_time, thread_res = time_threadpool(task_list, max_workers=tasks)
#     print(f"Threading:  {thread_time:.4f} seconds")

#     # Multiprocessing
#     proc_workers = min(os.cpu_count() or 2, tasks)
#     proc_time, proc_res = time_processpool(task_list, max_workers=proc_workers)
#     print(f"Multiprocessing ({proc_workers} workers): {proc_time:.4f} seconds")

#     # Correctness check
#     print("\nResults identical across all methods:",
#           seq_res == thread_res == proc_res)

####################################
# Parrallelism benchmark IO-bound
####################################

import time
import os
from concurrent.futures import ThreadPoolExecutor, ProcessPoolExecutor


def read_file(task_id: int, path: str) -> int:
    """Reads an entire file and returns the number of bytes read."""
    with open(path, "rb") as f:
        data = f.read()
    return len(data)

def bench_sequential(n_tasks, path):
    start = time.perf_counter()
    results = [read_file(i, path) for i in range(n_tasks)]
    return time.perf_counter() - start, results

def bench_threadpool(n_tasks, path, max_workers):
    start = time.perf_counter()
    with ThreadPoolExecutor(max_workers=max_workers) as ex:
        futures = [ex.submit(read_file, i, path) for i in range(n_tasks)]
        results = [f.result() for f in futures]
    return time.perf_counter() - start, results

def bench_processpool(n_tasks, path, max_workers):
    start = time.perf_counter()
    # map requires iterable arguments
    with ProcessPoolExecutor(max_workers=max_workers) as ex:
        # Provide both arguments via map
        results = list(ex.map(read_file,
                              range(n_tasks),        # task_id
                              [path] * n_tasks))     # file path
    return time.perf_counter() - start, results


# if __name__ == "__main__":
#     FILE_PATH = "testfile.dat"   # change this to any large file on disk
#     N_TASKS = 50                 # number of reads
#     THREAD_WORKERS = 8
#     PROCESS_WORKERS = 8

#     FILE_PATH = "testfile.dat"
#     SIZE_MB = 200   # change this to any size you want
#     CHUNK = 1024 * 1024  # 1 MB per write

#     print(f"Creating {SIZE_MB} MB file: {FILE_PATH}")

#     with open(FILE_PATH, "wb") as f:
#         for _ in range(SIZE_MB):
#             f.write(os.urandom(CHUNK))

#     if not os.path.exists(FILE_PATH):
#         raise FileNotFoundError(
#             f"File '{FILE_PATH}' does not exist. "
#             "Place a file with this name or edit FILE_PATH."
#         )

#     print(f"Using file: {FILE_PATH} ({os.path.getsize(FILE_PATH)} bytes)")
#     print(f"Tasks: {N_TASKS}")
#     print(f"Threads: {THREAD_WORKERS}, Processes: {PROCESS_WORKERS}\n")

#     # Run benchmarks
#     seq_time, seq_res = bench_sequential(N_TASKS, FILE_PATH)
#     print(f"Sequential: {seq_time:.4f} seconds")

#     th_time, th_res = bench_threadpool(N_TASKS, FILE_PATH, THREAD_WORKERS)
#     print(f"ThreadPoolExecutor ({THREAD_WORKERS} workers): {th_time:.4f} seconds")

#     proc_time, proc_res = bench_processpool(N_TASKS, FILE_PATH, PROCESS_WORKERS)
#     print(f"ProcessPoolExecutor ({PROCESS_WORKERS} workers): {proc_time:.4f} seconds")

#     print("\nResults (bytes read): seq / threads / processes")
#     print(seq_res[0], "/", th_res[0], "/", proc_res[0])
